// SPDX-License-Identifier: GPL-2.0
/*
 * Copyright (c) 2021-2022 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 */

#include <linux/module.h>
#include <linux/kobject.h>

#include <linux/gpio.h>
#include <linux/platform_device.h>
#include <linux/of_device.h>
#include <linux/of_platform.h>
#include <linux/delay.h>
#include <linux/types.h>

#define MODULE_NAME "icr"
#define ICR_DEVICE_GET_ATTR(_name) \
	(&dev_attr_##_name.attr)

/* Module parameter set by the RTOS if the day/night mode has been set. */
static char *rtos_mode;

/*
 * All ICR chips operate very much in the same way: They have 2 input pins which
 * control 2 output pins. For PWM input interface, the input pins are IN1 and IN2.
 * For PHASE/ENABLE interface, the input pins are PH and EN.
 * Some chips also have a SLEEP pin, while others have Vcc, but in both cases
 * there's a pin to enable/disable the chip.
 *
 * All PWM-based chips we used so far - DRV8210P, SY6712ABC and TMI8113 have the exact
 * same state table:
 *
 *   IN1  IN2  OUT1 OUT2  Function
 *    0    0    Z    Z     Coast
 *    0    1    L    H     Reverse
 *    1    0    H    L     Forward
 *    1    1    L    L     Brake
 *
 * We have the G2056ARC1U and DRV8838DSGR which use the PHASE/ENABLE interface.
 * The state table as below:
 *
 *   PH   EN   OUT1 OUT2  Function
 *    x    x    Z    Z     Coast
 *    1    1    L    H     Reverse
 *    0    1    H    L     Forward
 *    x    0    L    L     Brake
 *
 * Notice that Forward/Reverse may not correspond directly to DAY/NIGHT.
 * This depends on how exactly the filter is connected to the driver chip, and
 * on the filter's orientation.
 *
 * This driver assumes that Forward==DAY and Reverse==NIGHT.
 * To support products where it is the exact opposite simply swap the input
 * GPIOs in the device's DTS file.
 */

struct ircut_state {
	u32 ircut_out1;
	u32 ircut_out2;
};

struct icr_dev {
	struct gpio_desc *ircut_gpios[2];
	u32 wake_delay_us;
	u32 sleep_delay_us;
	u32 drive_duration_ms;
	u32 brake_delay_ms;
	u32 input_interface;
};

enum ircut_input_interface_e {
	INPUT_INTERFACE_PWM,
	INPUT_INTERFACE_PHASE_ENABLE
};

/* Raw ICR filter driving states */
enum ircut_state_e {
	ICR_STATE_COAST,
	ICR_STATE_REVERSE,
	ICR_STATE_FORWARD,
	ICR_STATE_BRAKE
};

static const struct ircut_state filt_states[][4] = {
	// PWM input interface
	{
		{0, 0}, // ICR_STATE_COAST
		{0, 1}, // ICR_STATE_REVERSE
		{1, 0}, // ICR_STATE_FORWARD
		{1, 1}  // ICR_STATE_BRAKE
	},
	// PHASE/ENABLE interface
	{
		{0, 0}, // ICR_STATE_COAST
		{1, 1}, // ICR_STATE_REVERSE
		{0, 1}, // ICR_STATE_FORWARD
		{1, 0}  // ICR_STATE_BRAKE
	}
};

static const char * const ircut_state_names[] = {
	"COAST",
	"REVERSE",
	"FORWARD",
	"BRAKE"
};

/* ICR modes; follow the data sheet recommended driving method.
 * Only DAY or NIGHT are valid modes.
 */
enum ircut_mode_e {
	ICR_MODE_UNKNOWN,
	ICR_MODE_NIGHT,
	ICR_MODE_DAY
};

static const char * const ircut_mode_names[] = {
	"UNKNOWN",
	"NIGHT",
	"DAY"
};

static int current_ircut_mode;

static const struct of_device_id icr_of_ids[] = {
	{ .compatible = "amazon,icr_cutfilter", },
	{ },
};
MODULE_DEVICE_TABLE(of, icr_of_ids);

/* From DRV8210P data sheet;  tWAKE: Sleep mode to active mode delay = 100us */
#define ICR_DEFAULT_WAKE_DELAY_US 100
/* From CM - apply delay before sleep also. */
#define ICR_DEFAULT_SLEEP_DELAY_US 100
/* From cut filter data sheet - drive for 100-160ms */
#define ICR_DEFAULT_DRIVE_DURATION_MS 150
/* From CM - apply 1 ms delay after BRAKE */
#define ICR_DEFAULT_BRAKE_DELAY_MS 1

void power_control_acquire(void);
void power_control_release(void);

static int set_icr_state(struct platform_device *pdev, int state)
{
	struct icr_dev *icrp = platform_get_drvdata(pdev);
	int ret = 0;

	if (state >= ARRAY_SIZE(filt_states[INPUT_INTERFACE_PWM]))
		return -EINVAL;

	gpiod_set_value(icrp->ircut_gpios[0], filt_states[icrp->input_interface][state].ircut_out1);
	gpiod_set_value(icrp->ircut_gpios[1], filt_states[icrp->input_interface][state].ircut_out2);

	return ret;
}

static int get_icr_state(struct platform_device *pdev)
{
	struct icr_dev *icrp = platform_get_drvdata(pdev);
	int icr0 = gpiod_get_value(icrp->ircut_gpios[0]);
	int icr1 = gpiod_get_value(icrp->ircut_gpios[1]);
	int i;
	int state = -EINVAL;

	for (i = 0; i < ARRAY_SIZE(filt_states[INPUT_INTERFACE_PWM]); i++) {
		if (filt_states[icrp->input_interface][i].ircut_out1 == icr0 &&
		    filt_states[icrp->input_interface][i].ircut_out2 == icr1) {
			state = i;
			break;
		}
	}
	return state;
}

/*
 * sysfs entries
 */

/*
 * NOTICE: The `icr_state_name` interface should only be used for factory
 * testing, and not for normal ICR operation.
 * Setting any state using this interface only guarantees that the INPUT lines
 * to the ICR will be set in the proper state.
 * The ICR OUTPUT pins might not reflect the new state because the ICR might be
 * disabled.
 *
 * To properly change the ICR state use the `icr_mode` interface instead.
 */
static ssize_t
icr_state_name_show(struct device *dev, struct device_attribute *attr,
		    char *buf)
{
	struct platform_device *pdev = to_platform_device(dev);
	int icr_state_num = get_icr_state(pdev);

	if (icr_state_num < 0)
		return icr_state_num;

	return scnprintf(buf, PAGE_SIZE, "%s\n",
			 ircut_state_names[icr_state_num]);
}

static ssize_t
icr_state_name_store(struct device *dev, struct device_attribute *attr,
		     const char *buf, size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	int ret = -EINVAL;
	int i;

	for (i = 0; i < ARRAY_SIZE(ircut_state_names); i++) {
		if (strncasecmp(ircut_state_names[i], buf,
				strlen(ircut_state_names[i])) == 0) {
			dev_info(&pdev->dev, "%s set state %-5s\n",
				 __func__, ircut_state_names[i]);
			ret = set_icr_state(pdev, i);
			break;
		}
	}

	if (ret == 0)
		return count;

	return ret;
}
static DEVICE_ATTR_RW(icr_state_name);

static int icr_mode_set(struct platform_device *pdev, int mode)
{
	int ret = -EINVAL;
	struct icr_dev *icrp = platform_get_drvdata(pdev);

	/* wake driver if necessary */
	power_control_acquire();
	udelay(icrp->wake_delay_us);

	switch (mode) {
	case ICR_MODE_DAY:
		ret = set_icr_state(pdev, ICR_STATE_FORWARD);
		if (ret != 0)
			goto exit;
		mdelay(icrp->drive_duration_ms);
		ret = set_icr_state(pdev, ICR_STATE_BRAKE);
		if (ret != 0)
			goto exit;
		mdelay(icrp->brake_delay_ms);
		ret = set_icr_state(pdev, ICR_STATE_COAST);
		break;
	case ICR_MODE_NIGHT:
		ret = set_icr_state(pdev, ICR_STATE_REVERSE);
		if (ret != 0)
			goto exit;
		mdelay(icrp->drive_duration_ms);
		ret = set_icr_state(pdev, ICR_STATE_BRAKE);
		if (ret != 0)
			goto exit;
		mdelay(icrp->brake_delay_ms);
		ret = set_icr_state(pdev, ICR_STATE_COAST);
		break;
	case ICR_MODE_UNKNOWN:
	default:
		goto exit;
	}
exit:
	/* Release driver power */
	power_control_release();
	udelay(icrp->sleep_delay_us);
	return ret;
}

static ssize_t
icr_mode_show(struct device *dev, struct device_attribute *attr,
	      char *buf)
{
	return scnprintf(buf, PAGE_SIZE, "%s\n",
			 ircut_mode_names[current_ircut_mode]);
}

static ssize_t
icr_mode_store(struct device *dev, struct device_attribute *attr,
	       const char *buf, size_t count)
{
	struct platform_device *pdev = to_platform_device(dev);
	int ret = -EINVAL;
	int i;
	int mode = -1;

	for (i = 0; i < ARRAY_SIZE(ircut_mode_names); i++) {
		if (strncasecmp(ircut_mode_names[i], buf,
				strlen(ircut_mode_names[i])) == 0) {
			mode = i;
			break;
		}
	}

	ret = icr_mode_set(pdev, mode);

	if (ret == 0) {
		dev_info(&pdev->dev, "%s set mode %-5s\n", __func__,
			 ircut_mode_names[mode]);
		current_ircut_mode = mode;
		return count;
	}

	/* error case */
	current_ircut_mode = ICR_MODE_UNKNOWN;
	return ret;
}
static DEVICE_ATTR_RW(icr_mode);

static struct attribute *icr_attrs[] = {
	ICR_DEVICE_GET_ATTR(icr_state_name),
	ICR_DEVICE_GET_ATTR(icr_mode),
	NULL
};

ATTRIBUTE_GROUPS(icr);

static int icr_probe(struct platform_device *pdev)
{
	struct icr_dev *icrp;
	struct device *dev = &pdev->dev;
	struct device_node *node = dev->of_node;
	int i, ret = 0;

	dev_dbg(dev, "%s\n", __func__);
	icrp = devm_kzalloc(dev, sizeof(*icrp), GFP_KERNEL);
	if (!icrp)
		return -ENOMEM;

	platform_set_drvdata(pdev, icrp);

	for (i = 0; i < ARRAY_SIZE(icrp->ircut_gpios); i++) {
		icrp->ircut_gpios[i] = devm_gpiod_get_index(dev, "icr",
							    i, GPIOD_OUT_LOW);
		if (IS_ERR(icrp->ircut_gpios[i])) {
			ret = PTR_ERR(icrp->ircut_gpios[i]);
			if (ret == -EPROBE_DEFER) {
				dev_info(dev,
					 "GPIOs not ready - defer probe\n");
				return ret;
			}
			goto err_exit;
		}
	}

	if (of_property_read_u32(node, "sleep_delay_us", &(icrp->sleep_delay_us))) {
		dev_warn(dev, "No sleep_delay_us - using default: %u\n", ICR_DEFAULT_SLEEP_DELAY_US);
		icrp->sleep_delay_us = ICR_DEFAULT_SLEEP_DELAY_US;
	}

	if (of_property_read_u32(node, "wake_delay_us", &(icrp->wake_delay_us))) {
		dev_warn(dev, "No wake_delay_us - using default: %u\n", ICR_DEFAULT_WAKE_DELAY_US);
		icrp->wake_delay_us = ICR_DEFAULT_WAKE_DELAY_US;
	}

	if (of_property_read_u32(node, "drive_duration_ms", &(icrp->drive_duration_ms))) {
		dev_warn(dev, "No drive_duration_ms - using default: %u\n", ICR_DEFAULT_DRIVE_DURATION_MS);
		icrp->drive_duration_ms = ICR_DEFAULT_DRIVE_DURATION_MS;
	}

	if (of_property_read_u32(node, "brake_delay_ms", &(icrp->brake_delay_ms))) {
		dev_warn(dev, "No brake_delay_ms - using default: %u\n", ICR_DEFAULT_BRAKE_DELAY_MS);
		icrp->brake_delay_ms = ICR_DEFAULT_BRAKE_DELAY_MS;
	}

	if (of_property_read_u32(node, "input_interface", &(icrp->input_interface))) {
		dev_warn(dev, "No input_interface - using default: PWM interface\n");
		icrp->input_interface = INPUT_INTERFACE_PWM;
	}

	ret = devm_device_add_groups(dev, icr_groups);
	if (ret) {
		dev_err(dev, "sysfs creation failed\n");
		goto err_exit;
	}

	if (rtos_mode) {
		if (strcmp(rtos_mode, "NIGHT") == 0)
			current_ircut_mode = ICR_MODE_NIGHT;
		else if (strcmp(rtos_mode, "DAY") == 0)
			current_ircut_mode = ICR_MODE_DAY;
	}

	dev_dbg(dev, "registered successfully\n");

	return 0;

err_exit:
	dev_err(dev, "err %d\n", ret);
	return ret;
}

static int icr_remove(struct platform_device *pdev)
{
	dev_dbg(&pdev->dev, "%s driver unloaded\n", pdev->name);
	return 0;
}

static struct platform_driver icr_driver = {
	.probe		= icr_probe,
	.remove		= icr_remove,
	.driver = {
		.name	= MODULE_NAME,
		.of_match_table = icr_of_ids,
		.suppress_bind_attrs = true,
	}
};
module_platform_driver(icr_driver);

/* Make these parameters directly available in sysfs */
module_param(rtos_mode, charp, 0400);
MODULE_PARM_DESC(rtos_mode, "Day/Night setting previously done by RTOS");

MODULE_AUTHOR("Richard Simmons <rssimmo@amazon.com>");
MODULE_AUTHOR("Avi Shukron <ashukro@amazon.com>");
MODULE_DESCRIPTION("IR cut filter driver");
MODULE_LICENSE("GPL v2");
